%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Function name: apply_co2_spec_int_corr
%
% Scope:    apply CO2 spectral interference correction
%
% Input:    data (structure),
%           bindata (structure),
%           binref (structure),
%           list of gas names (cell),
%           list of gas type "flags" (i.e., MPV positions) (double),
%           slope spectral interference correction (double)
%           anchor gas name (string),
%           index plot requested (double)
%
% Output:   corrected "bindata" and "binref" structures,
%           overview figures (if requested)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [ref, binref, bindata] = apply_co2_spec_int_corr(data, bindata, binref, gasnames, gastypeflag, slope_co2, anchor_gas, plot_requested)

tic;fprintf('Running function apply_co2_spec_int_corr...\n');

bindata_idx=2;     %N2O index
bindata_idx2=14;   %CO2 index
target_idx=4:6;     %d15N, d15Na, d15Nb index

ind_anchor=find(ismember(gasnames, anchor_gas));

%calculate intercept for fixed slopes (only needed for plotting, not used in correction)
y_anchor=NaN(1,width(target_idx));
intercept_co2=NaN(1,width(target_idx));
for z=1:width(target_idx)
    y_anchor(1,z) = slope_co2(1,z).*(binref.(gasnames{1,ind_anchor})(1,bindata_idx2)./binref.(gasnames{1,ind_anchor})(1,bindata_idx));
    intercept_co2(1,z)=binref.(gasnames{1,ind_anchor})(1,target_idx(1,z)) - y_anchor(1,z);
end

axislabel={'CO_2 / N_2O [ppm ppb^{-1}]'};

%% plot data before correction

if plot_requested
    plot_co2_spec_int_corr(data,bindata,binref,gasnames,bindata_idx,bindata_idx2,target_idx,slope_co2,intercept_co2,axislabel,ind_anchor,'Before CO_2 spectral interference correction');
end
%% apply correction

%correct data
for j=1:width(target_idx)
    for i=1:width(gastypeflag)
           bindata.(gasnames{1,i})(:,target_idx(1,j))=bindata.(gasnames{1,i})(:,target_idx(1,j))-slope_co2(1,j)*((bindata.(gasnames{1,i})(:,bindata_idx2)./bindata.(gasnames{1,i})(:,bindata_idx))-(binref.(gasnames{1,ind_anchor})(1,bindata_idx2)./binref.(gasnames{1,ind_anchor})(1,bindata_idx)));
    end
    bindata.all(:,target_idx(1,j))=bindata.all(:,target_idx(1,j))-slope_co2(1,j)*((bindata.all(:,bindata_idx2)./bindata.all(:,bindata_idx))-(binref.(gasnames{1,ind_anchor})(1,bindata_idx2)./binref.(gasnames{1,ind_anchor})(1,bindata_idx)));
end

%% recalculate ref conditions

[ref, binref, bindata] = calc_ref_conditions(data,bindata,gasnames,gastypeflag);

%% plot data after correction

if plot_requested
    plot_co2_spec_int_corr(data,bindata,binref,gasnames,bindata_idx,bindata_idx2,target_idx,slope_co2,intercept_co2,axislabel,ind_anchor,'After CO_2 spectral interference correction');
end

%%
time_elapsed=toc; fprintf('apply_co2_spec_int_corr completed (execution time: %1.2f s)\n',time_elapsed); 

end

